// Emacs style mode select   -*- C++ -*-
//-----------------------------------------------------------------------------
//
// Copyright(C) 2000 James Haley
//
// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
//--------------------------------------------------------------------------
//
// DESCRIPTION:
//      The actual column drawing functions.
//      Here find the main potential for optimization,
//       e.g. inline assembly, different algorithms.
//
//-----------------------------------------------------------------------------

static const char
rcsid[] = "$Id: r_draw.c,v 1.16 1998/05/03 22:41:46 killough Exp $";

#include "doomstat.h"
#include "w_wad.h"
#include "r_draw.h"
#include "r_main.h"
#include "v_video.h"
#include "mn_engin.h"
#include "d_gi.h"

#define MAXWIDTH  MAX_SCREENWIDTH          /* kilough 2/8/98 */
#define MAXHEIGHT MAX_SCREENHEIGHT

#ifdef DJGPP
#define USEASM /* sf: changed #ifdef DJGPP to #ifdef USEASM */
#endif

//
// All drawing to the view buffer is accomplished in this file.
// The other refresh files only know about ccordinates,
//  not the architecture of the frame buffer.
// Conveniently, the frame buffer is a linear one,
//  and we need only the base address,
//  and the total size == width*height*depth/8.,
//

byte *viewimage; 
int  viewwidth;
int  scaledviewwidth;
int  scaledviewheight;        // killough 11/98
int  viewheight;
int  viewwindowx;
int  viewwindowy; 
// SoM: ANYRES
int  scaledwindowx;
int  scaledwindowy;

byte *ylookup[MAXHEIGHT]; 
int  columnofs[MAXWIDTH]; 
int  linesize = SCREENWIDTH;  // killough 11/98

// Color tables for different players,
//  translate a limited part to another
//  (color ramps used for  suit colors).
//
 
byte *tranmap;          // translucency filter maps 256x256   // phares 
byte *main_tranmap;     // killough 4/11/98

//
// R_DrawColumn
// Source is the top of the column to scale.
//

lighttable_t *dc_colormap; 
int     dc_x; 
int     dc_yl; 
int     dc_yh; 
fixed_t dc_iscale; 
fixed_t dc_texturemid;
int     dc_texheight;    // killough
byte    *dc_source;      // first pixel in a column (possibly virtual) 
fixed_t dc_translevel;   // haleyjd: level for zdoom translucency


// Fuzz stuffs

const int fuzzoffset[FUZZTABLE] = 
{
  1,0,1,0,1,1,0,
  1,1,0,1,1,1,0,
  1,1,1,0,0,0,0,
  1,0,0,1,1,1,1,0,
  1,0,1,1,0,0,1,
  1,0,0,0,0,1,1,
  1,1,0,1,1,0,1 
}; 

int fuzzpos = 0; 

//
// A column is a vertical slice/span from a wall texture that,
//  given the DOOM style restrictions on the view orientation,
//  will always have constant z depth.
// Thus a special case loop for very fast rendering can
//  be used. It has also been used with Wolfenstein 3D.
// 

// haleyjd 04/10/04: FIXME -- ASM version of R_DrawColumn is out
// of sync currently.

//#ifndef USEASM     // killough 2/15/98

static void R_DrawColumn(void) 
{ 
   int              count; 
   register byte    *dest;            // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;     

   count = dc_yh - dc_yl + 1; 

   if(count <= 0)    // Zero length, column does not exceed a pixel.
      return;

#ifdef RANGECHECK 
   if(dc_x < 0 || dc_x >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);    
#endif 

   // Framebuffer destination address.
   // Use ylookup LUT to avoid multiply with ScreenWidth.
   // Use columnofs LUT for subwindows

   dest = ylookup[dc_yl] + columnofs[dc_x];

   // Determine scaling, which is the only mapping to be done.

   fracstep = dc_iscale; 
   frac = dc_texturemid + 
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   // Inner loop that does the actual texture mapping,
   //  e.g. a DDA-lile scaling.
   // This is as fast as it gets.       (Yeah, right!!! -- killough)
   //
   // killough 2/1/98: more performance tuning
   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
          
         if(frac < 0)
            while((frac += heightmask) <  0);
         else
            while(frac >= heightmask)
               frac -= heightmask;
          
         do
         {
            // Re-map color indices from wall texture column
            //  using a lighting/special effects LUT.
            
            // heightmask is the Tutti-Frutti fix -- killough
            
            *dest = colormap[source[frac>>FRACBITS]];
            dest += linesize;                     // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
            dest += linesize;   // killough 11/98
            frac += fracstep;
            *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
            *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
      }
   }   
} 

// haleyjd 04/10/04: FIXME -- ASM version of R_DrawColumn is out
// of sync currently.

//#endif

// Here is the version of R_DrawColumn that deals with translucent  // phares
// textures and sprites. It's identical to R_DrawColumn except      //    |
// for the spot where the color index is stuffed into *dest. At     //    V
// that point, the existing color index and the new color index
// are mapped through the TRANMAP lump filters to get a new color
// index whose RGB values are the average of the existing and new
// colors.
//
// Since we're concerned about performance, the 'translucent or
// opaque' decision is made outside this routine, not down where the
// actual code differences are.

// haleyjd 04/10/04: FIXME -- ASM version of R_DrawTLColumn is out
// of sync currently.

//#ifndef USEASM                       // killough 2/21/98: converted to x86 asm

static void R_DrawTLColumn(void)
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   
   count = dc_yh - dc_yl + 1; 
   
   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x < 0 || dc_x >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawTLColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);    
#endif 

   // Framebuffer destination address.
   // Use ylookup LUT to avoid multiply with ScreenWidth.
   // Use columnofs LUT for subwindows? 
   
   dest = ylookup[dc_yl] + columnofs[dc_x];
   
   // Determine scaling,
   //  which is the only mapping to be done.
   
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   // Inner loop that does the actual texture mapping,
   //  e.g. a DDA-lile scaling.
   // This is as fast as it gets.       (Yeah, right!!! -- killough)
   //
   // killough 2/1/98, 2/21/98: more performance tuning
   
   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
         
         if(frac < 0)
            while((frac += heightmask) <  0);
         else
            while(frac >= heightmask)
               frac -= heightmask;
        
         do
         {
            // Re-map color indices from wall texture column
            //  using a lighting/special effects LUT.
            
            // heightmask is the Tutti-Frutti fix -- killough
            
            *dest = tranmap[(*dest<<8) + colormap[source[frac>>FRACBITS]]]; // phares
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            *dest = tranmap[(*dest<<8)+colormap[source[(frac>>FRACBITS) & heightmask]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
            *dest = tranmap[(*dest<<8)+colormap[source[(frac>>FRACBITS) & heightmask]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
            *dest = tranmap[(*dest<<8)+colormap[source[(frac>>FRACBITS) & heightmask]]]; // phares
      }
   }   
} 

// haleyjd 04/10/04: FIXME -- ASM version of R_DrawTLColumn is out
// of sync currently.

//#endif  // killough 2/21/98: converted to x86 asm

//
// haleyjd 02/08/05: BOOM TL/Tlated was neglected.
//
static void R_DrawTLTRColumn(void)
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   
   count = dc_yh - dc_yl + 1; 
   
   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x < 0 || dc_x >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawTLTRColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);    
#endif 

   // Framebuffer destination address.
   // Use ylookup LUT to avoid multiply with ScreenWidth.
   // Use columnofs LUT for subwindows? 
   
   dest = ylookup[dc_yl] + columnofs[dc_x];
   
   // Determine scaling,
   //  which is the only mapping to be done.
   
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   // Inner loop that does the actual texture mapping,
   //  e.g. a DDA-lile scaling.
   // This is as fast as it gets.       (Yeah, right!!! -- killough)
   //
   // killough 2/1/98, 2/21/98: more performance tuning
   
   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
         
         if(frac < 0)
            while((frac += heightmask) <  0);
         else
            while(frac >= heightmask)
               frac -= heightmask;
        
         do
         {
            // Re-map color indices from wall texture column
            //  using a lighting/special effects LUT.
            
            // heightmask is the Tutti-Frutti fix -- killough
            
            *dest = tranmap[(*dest<<8) + colormap[dc_translation[source[frac>>FRACBITS]]]]; // phares
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            *dest = tranmap[(*dest<<8)+colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
            *dest = tranmap[(*dest<<8)+colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
            *dest = tranmap[(*dest<<8)+colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]]; // phares
      }
   }   
} 

//
// Spectre/Invisibility.
//


// SoM: Fuzz Stuff moved to beginning of the file
//
// Framebuffer postprocessing.
// Creates a fuzzy image by copying pixels
//  from adjacent ones to left and right.
// Used with an all black colormap, this
//  could create the SHADOW effect,
//  i.e. spectres and invisible players.
//

// sf: restored original fuzz effect (changed in mbf)
// sf: changed to use vis->colormap not fullcolormap
//     for coloured lighting and SHADOW now done with
//     flags not NULL colormap

static void R_DrawFuzzColumn(void) 
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;

   // Adjust borders. Low...
   if(!dc_yl) 
      dc_yl = 1;
   
   // .. and high.
   if(dc_yh == viewheight - 1) 
      dc_yh = viewheight - 2; 

   count = dc_yh - dc_yl; 

    // Zero length.
    if(count < 0) 
	return; 
       
#ifdef RANGECHECK 
   // haleyjd: these should apparently be adjusted for hires
   // SoM: DONE
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawFuzzColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif

   // Keep till detailshift bug in blocky mode fixed,
   //  or blocky mode removed.
   
   // Does not work with blocky mode.   
   dest = ylookup[dc_yl] + columnofs[dc_x];
   
   // Looks familiar.
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);
   
   // Looks like an attempt at dithering,
   // using the colormap #6 (of 0-31, a bit brighter than average).
   
   do 
   {
      // Lookup framebuffer, and retrieve
      //  a pixel that is either one column
      //  left or right of the current one.
      // Add index from colormap to index.
      // killough 3/20/98: use fullcolormap instead of colormaps
      // sf: use dc_colormap for coloured lighting
      
      //sf : hires
      *dest = dc_colormap[6*256+dest[fuzzoffset[fuzzpos] ? v_width : -v_width]];
      
      // Clamp table lookup index.
      if(++fuzzpos == FUZZTABLE) 
         fuzzpos = 0;
      
      dest += linesize;
      frac += fracstep; 
   } 
   while(count--);
}

//
// R_DrawTranslatedColumn
// Used to draw player sprites
//  with the green colorramp mapped to others.
// Could be used with different translation
//  tables, e.g. the lighter colored version
//  of the BaronOfHell, the HellKnight, uses
//  identical sprites, kinda brightened up.
//

// haleyjd: changed translationtables to byte **
byte *dc_translation, **translationtables = NULL;

// haleyjd: new stuff
int firsttranslationlump, lasttranslationlump;
int numtranslations = 0;

static void R_DrawTRColumn(void) 
{ 
   int      count; 
   byte     *dest; 
   fixed_t  frac;
   fixed_t  fracstep;     
   
   count = dc_yh - dc_yl + 1; 
   if(count <= 0) 
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawTRColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif 

   dest = ylookup[dc_yl] + columnofs[dc_x];
   
   // Looks familiar.
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   // Here we do an additional index re-mapping.
   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
         
         if(frac < 0)
            while ((frac += heightmask) <  0);
         else
            while (frac >= (int)heightmask)
               frac -= heightmask;   
         do
         {
            *dest = colormap[dc_translation[source[frac>>FRACBITS]]]; // phares
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            *dest = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
            *dest = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]; // phares
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
            *dest = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]; // phares
      }
   }
} 

//
// R_DrawFlexTLColumn
//
// haleyjd 09/01/02: zdoom-style translucency
//
static void R_DrawFlexColumn(void)
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   unsigned int *fg2rgb, *bg2rgb;
   register unsigned int fg, bg;
   
   count = dc_yh - dc_yl + 1; 

   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawFlexColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif 

   {
      fixed_t fglevel, bglevel;
      
      fglevel = dc_translevel & ~0x3ff;
      bglevel = FRACUNIT - fglevel;
      fg2rgb  = Col2RGB[fglevel >> 10];
      bg2rgb  = Col2RGB[bglevel >> 10];
   }

   dest = ylookup[dc_yl] + columnofs[dc_x];
  
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if (dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
          
         if (frac < 0)
            while ((frac += heightmask) <  0);
         else
            while (frac >= (int)heightmask)
               frac -= heightmask;          

         do
         {
            fg = colormap[source[frac>>FRACBITS]];
            bg = *dest;

            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;
            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0)   // texture height is a power of 2 -- killough
         {
            fg = colormap[source[(frac>>FRACBITS) & heightmask]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;
            
            fg = colormap[source[(frac>>FRACBITS) & heightmask]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
         {
            fg = colormap[source[(frac>>FRACBITS) & heightmask]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
         }
      }
   }
}

//
// R_DrawFlexTlatedColumn
//
// haleyjd 11/05/02: zdoom-style translucency w/translation, for
// player sprites
//
static void R_DrawFlexTRColumn(void) 
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   unsigned int *fg2rgb, *bg2rgb;
   register unsigned int fg, bg;
   
   count = dc_yh - dc_yl + 1; 

   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawFlexTRColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif 

   {
      fixed_t fglevel, bglevel;
      
      fglevel = dc_translevel & ~0x3ff;
      bglevel = FRACUNIT - fglevel;
      fg2rgb  = Col2RGB[fglevel >> 10];
      bg2rgb  = Col2RGB[bglevel >> 10];
   }

   dest = ylookup[dc_yl] + columnofs[dc_x];
  
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if (dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
          
         if (frac < 0)
            while ((frac += heightmask) <  0);
         else
            while (frac >= (int)heightmask)
               frac -= heightmask;          

         do
         {
            fg = colormap[dc_translation[source[frac>>FRACBITS]]];
            bg = *dest;

            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;
            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            fg = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;
            
            fg = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;
         }
         if(count & 1)
         {
            fg = colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]];
            bg = *dest;
            fg = fg2rgb[fg];
            bg = bg2rgb[bg];
            fg = (fg+bg) | 0xf07c3e1f;

            *dest = RGB8k[0][0][(fg>>5) & (fg>>19)];
         }
      }
   }
} 

//
// R_DrawAddColumn
//
// haleyjd 02/08/05: additive translucency
//
static void R_DrawAddColumn(void)
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   unsigned int *fg2rgb, *bg2rgb;
   unsigned int a, b;
   
   count = dc_yh - dc_yl + 1; 

   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawAddColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif 

   {
      fixed_t fglevel, bglevel;
      
      fglevel = dc_translevel & ~0x3ff;
      bglevel = FRACUNIT;
      fg2rgb  = Col2RGB[fglevel >> 10];
      bg2rgb  = Col2RGB[bglevel >> 10];
   }

   dest = ylookup[dc_yl] + columnofs[dc_x];
  
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
          
         if (frac < 0)
            while ((frac += heightmask) <  0);
         else
            while (frac >= (int)heightmask)
               frac -= heightmask;          

         do
         {
            // mask out LSBs in green and red to allow overflow
            a = fg2rgb[colormap[source[frac>>FRACBITS]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0)   // texture height is a power of 2 -- killough
         {
            a = fg2rgb[colormap[source[(frac>>FRACBITS) & heightmask]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;

            a = fg2rgb[colormap[source[(frac>>FRACBITS) & heightmask]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;            
         }
         if(count & 1)
         {
            a = fg2rgb[colormap[source[(frac>>FRACBITS) & heightmask]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
         }
      }
   }   
}

//
// R_DrawAddTlatedColumn
//
// haleyjd 02/08/05: additive translucency + translation
// The slowest of all column drawers!
//
static void R_DrawAddTRColumn(void) 
{ 
   int              count; 
   register byte    *dest;           // killough
   register fixed_t frac;            // killough
   fixed_t          fracstep;
   unsigned int *fg2rgb, *bg2rgb;
   unsigned int a, b;
   
   count = dc_yh - dc_yl + 1; 

   // Zero length, column does not exceed a pixel.
   if(count <= 0)
      return; 
                                 
#ifdef RANGECHECK 
   if(dc_x  < 0 || dc_x  >= v_width || dc_yl < 0 || dc_yh >= v_height)
      I_Error("R_DrawAddTRColumn: %i to %i at %i", dc_yl, dc_yh, dc_x);
#endif 

   {
      fixed_t fglevel, bglevel;
      
      fglevel = dc_translevel & ~0x3ff;
      bglevel = FRACUNIT;
      fg2rgb  = Col2RGB[fglevel >> 10];
      bg2rgb  = Col2RGB[bglevel >> 10];
   }

   dest = ylookup[dc_yl] + columnofs[dc_x];
  
   fracstep = dc_iscale; 
   frac = dc_texturemid +
      FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

   {
      register const byte *source = dc_source;            
      register const lighttable_t *colormap = dc_colormap; 
      register heightmask = dc_texheight-1;
      if(dc_texheight & heightmask)   // not a power of 2 -- killough
      {
         heightmask++;
         heightmask <<= FRACBITS;
          
         if (frac < 0)
            while ((frac += heightmask) <  0);
         else
            while (frac >= (int)heightmask)
               frac -= heightmask;          

         do
         {
            // mask out LSBs in green and red to allow overflow
            a = fg2rgb[colormap[dc_translation[source[frac>>FRACBITS]]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            
            dest += linesize;          // killough 11/98
            if((frac += fracstep) >= heightmask)
               frac -= heightmask;
         } 
         while(--count);
      }
      else
      {
         while((count -= 2) >= 0) // texture height is a power of 2 -- killough
         {
            a = fg2rgb[colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;

            a = fg2rgb[colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
            dest += linesize;   // killough 11/98
            frac += fracstep;            
         }
         if(count & 1)
         {
            a = fg2rgb[colormap[dc_translation[source[(frac>>FRACBITS) & heightmask]]]] & 0xFFBFDFF;
            b = bg2rgb[*dest] & 0xFFBFDFF;
            
            a  = a + b;                      // add with overflow
            b  = a & 0x10040200;             // isolate LSBs
            b  = (b - (b >> 5)) & 0xF83C1E0; // convert to clamped values
            a |= 0xF07C3E1F;                 // apply normal tl mask
            a |= b;                          // mask in clamped values
            
            *dest = RGB8k[0][0][(a >> 5) & (a >> 19)];
         }
      }
   }
} 

//
// Normal Column Drawer Object
// haleyjd 09/04/06
//
columndrawer_t r_normal_drawer =
{
   R_DrawColumn,
   R_DrawTLColumn,
   R_DrawTRColumn,
   R_DrawTLTRColumn,
   R_DrawFuzzColumn,
   R_DrawFlexColumn,
   R_DrawFlexTRColumn,
   R_DrawAddColumn,
   R_DrawAddTRColumn,

   NULL
};

//
// R_InitTranslationTables
// Creates the translation tables to map
//  the green color ramp to gray, brown, red.
// Assumes a given structure of the PLAYPAL.
// Could be read from a lump instead.
//

typedef struct
{
  int start;      // start of the sequence of colours
  int number;     // number of colours
} translat_t;

translat_t translations[TRANSLATIONCOLOURS] =
{
    {96,  16},     // indigo
    {64,  16},     // brown
    {32,  16},     // red
  
  /////////////////////////
  // New colours
  
    {176, 16},     // tomato
    {128, 16},     // dirt
    {200, 8},      // blue
    {160, 8},      // gold
    {152, 8},      // felt?
    {0,   1},      // bleeacckk!!
    {250, 5},      // purple
  //  {168, 8}, // bright pink, kinda
    {216, 8},      // vomit yellow
    {16,  16},     // pink
    {56,  8},      // cream
    {88,  8},      // white
};

// 
// R_InitTranslationTables
//
// haleyjd 01/12/04: rewritten to support translation lumps
//
void R_InitTranslationTables(void)
{
   int numlumps, i, c;
   
   // don't leak the allocation
   if(translationtables)
   {
      for(i = 0; i < numtranslations; ++i)
         Z_Free(translationtables[i]);

      Z_Free(translationtables);

      // SoM: let's try... this.
      translationtables = NULL;
   }

   // count number of lumps
   firsttranslationlump = W_CheckNumForName("T_START");
   lasttranslationlump  = W_CheckNumForName("T_END");

   if(firsttranslationlump == -1 || lasttranslationlump == -1)
      numlumps = 0;
   else
      numlumps = (lasttranslationlump - firsttranslationlump) - 1;

   // set numtranslations
   numtranslations = TRANSLATIONCOLOURS + numlumps;

   // allocate the array of pointers
   translationtables = Z_Malloc(sizeof(byte *) * numtranslations, PU_STATIC, 0);
   
   // build the internal player translations
   for(i = 0; i < TRANSLATIONCOLOURS; ++i)
   {
      byte *transtbl;

      transtbl = translationtables[i] = Z_Malloc(256, PU_STATIC, 0);

      for(c = 0; c < 256; ++c)
      {
         transtbl[c] =
            (c < 0x70 || c > 0x7f) ? c : translations[i].start +
             ((c & 0xf) * (translations[i].number-1))/15;
      }
   }

   // read in the lumps, if any
   for(i = TRANSLATIONCOLOURS; i < numtranslations; ++i)
   {
      int lumpnum = (i - TRANSLATIONCOLOURS) + firsttranslationlump + 1;

      translationtables[i] = W_CacheLumpNum(lumpnum, PU_STATIC);
   }
}



////////////////////////////////////////////////////////////////
// SoM: moved span drawers to r_span.c
////////////////////////////////////////////////////////////////

//
// R_InitBuffer 
// Creates lookup tables that avoid
//  multiplies and other hazzles
//  for getting the framebuffer address
//  of a pixel to draw.
//


// SoM: ANYRES
void R_InitBuffer(int width, int height)
{ 
   int i; 
   int st_height;
   int tviewwidth = viewwidth << detailshift;
   
   linesize = v_width;    // killough 11/98
   
   // Handle resize,
   //  e.g. smaller view windows
   //  with border and/or status bar.
   viewwindowx = (v_width-tviewwidth) >> 1;
   scaledwindowx = (SCREENWIDTH - width) >> 1;
   // Column offset. For windows.
   for (i = tviewwidth ; i--; )   // killough 11/98
      columnofs[i] = viewwindowx + i;
   
   // Same with base row offset.
   st_height = gameModeInfo->StatusBar->height;
   
   if(tviewwidth == v_width)
      viewwindowy = scaledwindowy = 0;
   else
   {
      viewwindowy = (v_height - ((st_height * globalyscale) >> FRACBITS) - viewheight) >> 1;
      scaledwindowy = (SCREENHEIGHT - st_height - height) >> 1;
   }
   
   // Precalculate all row offsets.
   
   for(i = viewheight; i--; )
      ylookup[i] = screens[0] + (i + viewwindowy) * linesize; // killough 11/98
} 

//
// R_FillBackScreen
// Fills the back screen with a pattern
//  for variable screen sizes
// Also draws a beveled edge.
//

void R_FillBackScreen (void) 
{ 
  // killough 11/98: trick to shadow variables
  // SoM: ANYRES use scaledwindowx and scaledwindowy instead
  int x, y; 
  patch_t *patch;

  giborder_t *border = gameModeInfo->border;

  int offset = border->offset;
  int size   = border->size;

  if(scaledviewwidth == 320)
    return;

  // haleyjd 08/16/02: some restructuring to use gameModeInfo

  // killough 11/98: use the function in m_menu.c
  V_DrawBackground(gameModeInfo->borderFlat, &backscreen1);
        
  patch = W_CacheLumpName(border->top, PU_CACHE);

  for(x = 0; x < scaledviewwidth; x += size)
    V_DrawPatch(scaledwindowx+x,scaledwindowy-offset,&backscreen1,patch);

  patch = W_CacheLumpName(border->bottom, PU_CACHE);

  for(x = 0; x < scaledviewwidth; x += size)   // killough 11/98:
    V_DrawPatch(scaledwindowx+x,scaledwindowy+scaledviewheight,&backscreen1,patch);

  patch = W_CacheLumpName(border->left, PU_CACHE);

  for(y = 0; y < scaledviewheight; y += size)  // killough 11/98
    V_DrawPatch(scaledwindowx-offset,scaledwindowy+y,&backscreen1,patch);
  
  patch = W_CacheLumpName(border->right, PU_CACHE);

  for(y = 0; y < scaledviewheight; y += size)  // killough 11/98
    V_DrawPatch(scaledwindowx+scaledviewwidth,scaledwindowy+y,&backscreen1,patch);

  // Draw beveled edge. 
  V_DrawPatch(scaledwindowx-offset,
              scaledwindowy-offset,
              &backscreen1,
              W_CacheLumpName(border->c_tl, PU_CACHE));
    
  V_DrawPatch(scaledwindowx+scaledviewwidth,
              scaledwindowy-offset,
              &backscreen1,
              W_CacheLumpName(border->c_tr, PU_CACHE));
    
  V_DrawPatch(scaledwindowx-offset,
              scaledwindowy+scaledviewheight,             // killough 11/98
              &backscreen1,
              W_CacheLumpName(border->c_bl, PU_CACHE));
    
  V_DrawPatch(scaledwindowx+scaledviewwidth,
              scaledwindowy+scaledviewheight,             // killough 11/98
              &backscreen1,
              W_CacheLumpName(border->c_br, PU_CACHE));
} 

//
// Copy a screen buffer.
//
// SoM: why the hell was this written to only take an offset and size parameter?
// this is a much nicer solution which fixes scaling issues in highres modes that aren't
// perfectly 4/3
//
void R_VideoErase(unsigned int x, unsigned int y, unsigned int w, unsigned int h)
{ 
   unsigned int ofs;

   // haleyjd 06/08/05: protect against bad erasings
   if(x + w > SCREENWIDTH || y + h > SCREENHEIGHT)
      return;

   // SoM: ANYRES
   // This receives scaled offsets.
   if(v_width != SCREENWIDTH || v_height != SCREENHEIGHT)
   {
      w = realxarray[x + w] - realxarray[x];
      h = realyarray[y + h] - realyarray[y];
      x = realxarray[x];
      y = realyarray[y];
   }
         
   ofs = x + y * v_width;

   if(x == 0 && w == (unsigned int)v_width)
   {
      memcpy(screens[0]+ofs, screens[1]+ofs, w * h);   // LFB copy.
      return;
   }

   ofs += (h - 1) * v_width;

   while(h-- > 0)
   {
      memcpy(screens[0] + ofs, screens[1] + ofs, w);
      ofs -= v_width;
   }
} 

//
// R_DrawViewBorder
// Draws the border around the view
//  for different size windows?
//
// SoM: Removed old killough hack and reformatted to use new R_VideoErase
//
void R_DrawViewBorder(void) 
{ 
   int side, st_height;
   
   if(scaledviewwidth == SCREENWIDTH) 
      return;

   // copy top
   // SoM: ANYRES
   R_VideoErase(0, 0, SCREENWIDTH, scaledwindowy);

   // copy sides
   side = scaledwindowx;
   R_VideoErase(0, scaledwindowy, side, scaledviewheight);
   R_VideoErase(SCREENWIDTH - side, scaledwindowy, side, scaledviewheight);

   // copy bottom 
   R_VideoErase(0, scaledwindowy + scaledviewheight, SCREENWIDTH, scaledwindowy);

   st_height = gameModeInfo->StatusBar->height;
   
   V_MarkRect(0,0,SCREENWIDTH, SCREENHEIGHT-st_height); 
} 

// haleyjd: experimental column drawer for masked sky textures
void R_DrawNewSkyColumn(void) 
{ 
  int              count; 
  register byte    *dest;            // killough
  register fixed_t frac;            // killough
  fixed_t          fracstep;     

  count = dc_yh - dc_yl + 1; 

  if (count <= 0)    // Zero length, column does not exceed a pixel.
    return; 
                                 
#ifdef RANGECHECK 
  if ((unsigned)dc_x >= MAX_SCREENWIDTH
      || dc_yl < 0
      || dc_yh >= MAX_SCREENHEIGHT) 
    I_Error ("R_DrawNewSkyColumn: %i to %i at %i", dc_yl, dc_yh, dc_x); 
#endif 

  // Framebuffer destination address.
  // Use ylookup LUT to avoid multiply with ScreenWidth.
  // Use columnofs LUT for subwindows? 

  dest = ylookup[dc_yl] + columnofs[dc_x];  

  // Determine scaling, which is the only mapping to be done.

  fracstep = dc_iscale; 
  //frac = dc_texturemid + (dc_yl-centery)*fracstep; 
  frac = dc_texturemid +
     FixedMul((dc_yl << FRACBITS) - centeryfrac, fracstep);

  // Inner loop that does the actual texture mapping,
  //  e.g. a DDA-lile scaling.
  // This is as fast as it gets.       (Yeah, right!!! -- killough)
  //
  // killough 2/1/98: more performance tuning

  {
    register const byte *source = dc_source;            
    register const lighttable_t *colormap = dc_colormap; 
    register int heightmask = dc_texheight-1;
    if (dc_texheight & heightmask)   // not a power of 2 -- killough
      {
        heightmask++;
        heightmask <<= FRACBITS;
          
        if (frac < 0)
          while ((frac += heightmask) <  0);
        else
          while (frac >= heightmask)
            frac -= heightmask;
          
        do
          {
            // Re-map color indices from wall texture column
            //  using a lighting/special effects LUT.
            
            // heightmask is the Tutti-Frutti fix -- killough

            // haleyjd
            if(source[frac>>FRACBITS])
              *dest = colormap[source[frac>>FRACBITS]];
            dest += linesize;                     // killough 11/98
            if ((frac += fracstep) >= heightmask)
              frac -= heightmask;
          } 
        while (--count);
      }
    else
      {
        while ((count-=2)>=0)   // texture height is a power of 2 -- killough
          {
            if(source[(frac>>FRACBITS) & heightmask])
              *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
            dest += linesize;   // killough 11/98
            frac += fracstep;
            if(source[(frac>>FRACBITS) & heightmask])
              *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
            dest += linesize;   // killough 11/98
            frac += fracstep;
          }
        if ((count & 1) && source[(frac>>FRACBITS) & heightmask])
          *dest = colormap[source[(frac>>FRACBITS) & heightmask]];
      }
  }
} 


//----------------------------------------------------------------------------
//
// $Log: r_draw.c,v $
// Revision 1.16  1998/05/03  22:41:46  killough
// beautification
//
// Revision 1.15  1998/04/19  01:16:48  killough
// Tidy up last fix's code
//
// Revision 1.14  1998/04/17  15:26:55  killough
// fix showstopper
//
// Revision 1.13  1998/04/12  01:57:51  killough
// Add main_tranmap
//
// Revision 1.12  1998/03/23  03:36:28  killough
// Use new 'fullcolormap' for fuzzy columns
//
// Revision 1.11  1998/02/23  04:54:59  killough
// #ifdef out translucency code since its in asm
//
// Revision 1.10  1998/02/20  21:57:04  phares
// Preliminarey sprite translucency
//
// Revision 1.9  1998/02/17  06:23:40  killough
// #ifdef out code duplicated in asm for djgpp targets
//
// Revision 1.8  1998/02/09  03:18:02  killough
// Change MAXWIDTH, MAXHEIGHT defintions
//
// Revision 1.7  1998/02/02  13:17:55  killough
// performance tuning
//
// Revision 1.6  1998/01/27  16:33:59  phares
// more testing
//
// Revision 1.5  1998/01/27  16:32:24  phares
// testing
//
// Revision 1.4  1998/01/27  15:56:58  phares
// Comment about invisibility
//
// Revision 1.3  1998/01/26  19:24:40  phares
// First rev with no ^Ms
//
// Revision 1.2  1998/01/26  05:05:55  killough
// Use unrolled version of R_DrawSpan
//
// Revision 1.1.1.1  1998/01/19  14:03:02  rand
// Lee's Jan 19 sources
//
//
//----------------------------------------------------------------------------
